% USE THIS INSTEAD OF plotDataFromVideo WHEN ANALYZING DATA GENERATED BY
% THE BEHAVIOR ANALYZER PACKAGE.
%
% EXAMPLE USAGE (see note below too):
% > plotMsDataFromVideo('', mock_pumps{1}, mock_pumps_miss, mock_rinses{1}, mock_rinses_miss, 0, 30, 1, 0, 1)
%
%NOTE: the bracket use in 'mock_pumps{1}' above does not work. You need to
%use mock_pumps(1,:), which gives the data as a cell array. I'm not sure
%what has changed, but I suspect that I might be saving the data in a
%slightly different way than Nikhil did -SS, 4/5/2017


% This function takes in a set of frame numbers that have been
% hand-annotated by watching videos of pumping, that indicate when the
% contraction occurs.  It then plots the data.
%
% Note that if the worm moves too quickly certain frames of the video may
% not include the pharynx.  These frames are indicated in the 'missing'
% variable, and dealt with as if they were missing data.  This seems like
% the most reasonable way to deal with this data.
%
% Both pumpFrames and missingFrames are cell arrays, with each cell
% containing the data from 1 worm.
%
% start is the time to start analyzing, in seconds. 
% stop is the time to stop analyzing, in seconds.
% winWidth is the number of seconds over which to do the average.
% (DIFFERENT than other scripts, where it is in milliseconds)

function plotMsDataFromVideoSpits(name, pumpMs, missingMs, pumpMs2, missingMs2, ...
    startInSec, stopInSec, winWidthInSec, normFlag, forpaper)

startDelay = 10;  % until stim on
stimLen = 10;
yMin = -.1;
yMax = 6.5;
eventName = 'pumping rate (Hz)';
yBar = 6;
numMsInS = 1000;
xStart = 1;  % Set to 1/numMsInS to plot forward moving average, 1 to plot backward moving average

lightc = [0 0 0] ./ 255 %[98 0 255] ./ 255; %BLUE %[0 0 0] ./ 255;  %VIOLET % Color of light bar
linecm = [%0 0 0; 
          %0 0 0; %black
%255 51 0; %(for orange)
          0  0 255; %blue
          0 100 0; %green
          0 0 255;
          ] ./ 255;  % Color of 2 mean lines
patchcm = [%171 171 171; 
           %171 171 171; %grey
           %255 214 126
          %255 214 126; %(for orange)
           215 215 255 %blue
           152 251 152 %light green
            ] ./ 255;  % Color of 2 error regions around the mean

winWidthInMs = floor(winWidthInSec * numMsInS);
startMs = floor(startInSec * numMsInS) + 1;  % Add one so a start of 0 gets converted to frame #1
stopMs = floor(stopInSec * numMsInS - winWidthInMs);
totalTime = stopInSec - startInSec;

% This is a cell vector, with each cell containing a vector where each
% value indicates the average pumping rate for that timepoint for one
% video.  This cell array will be analyze later to determine the mean and
% standard deviation to be plotted.
pumpRate = cell(1, stopMs - startMs + 2);

for ms = startMs:stopMs+1   % For each frame to analyze
    for vid = 1:length(pumpMs)   % For each video to analyze
        % If the grinder is not missing from this frame in the video, and
        % the whole window to be analyzed is clear of missing data
        if (isempty(find(missingMs{vid} >= ms & missingMs{vid} < ms+winWidthInMs, 1)))  
            rate = length(find(pumpMs{vid} >= ms & pumpMs{vid} < ms+winWidthInMs)) / winWidthInSec;
            pumpRate{ms-startMs+1} = cat(1, pumpRate{ms-startMs+1}, rate);
        end
    end
end

% Now that all the data is collated, calculate the mean pump rate over time
m = cellfun(@mean, pumpRate);
m(isnan(m)) = 0;
sd = cellfun(@std, pumpRate);
se = sd ./ sqrt(cellfun(@length, pumpRate));
se(isnan(se)) = 0;

if (~isempty(pumpMs2))
    pumpRate2 = cell(1, stopMs - startMs + 2);
    for ms = startMs:stopMs+1   % For each frame to analyze
        for vid = 1:length(pumpMs2)   % For each video to analyze
            % If the grinder is not missing from this frame in the video,, and
            % the whole window to be analyzed is clear of missing data
            if (isempty(find(missingMs2{vid} >= ms & missingMs2{vid} < ms+winWidthInMs, 1)))  
                rate = length(find(pumpMs2{vid} >= ms & pumpMs2{vid} < ms+winWidthInMs)) / winWidthInSec;
                pumpRate2{ms-startMs+1} = cat(1, pumpRate2{ms-startMs+1}, rate);
            end
        end
    end

    % Now that all the data is collated, calculate the mean pump rate over time
    m2 = cellfun(@mean, pumpRate2);
    m2(isnan(m2)) = 0;
    sd2 = cellfun(@std, pumpRate2);
    se2 = sd2 ./ sqrt(cellfun(@length, pumpRate2));
    se2(isnan(se2)) = 0;
end

if (normFlag)
    yMax = 1.8;
    s1norm = mean(m(1:startDelay*numMsInS));
    m = m/s1norm;
    sd = sd/s1norm;
    se = sd / sqrt(length(pumpMs));

    if (~isempty(pumpMs2))
        s2norm = mean(m2(1:startDelay*numMsInS));
        m2 = m2/s2norm;
        sd2 = sd2/s2norm;
        se2 = sd2 / sqrt(length(pumpMs2));
    end
end


figure;
hold on

x = xStart + (0:1/numMsInS:totalTime - winWidthInSec);

patch([x x(end:-1:1)], [m+se m(end:-1:1)-se(end:-1:1)], patchcm(1,:),'LineStyle','none');
if (~isempty(pumpMs2))
    patch([x x(end:-1:1)], [m2+se2 m2(end:-1:1)-se2(end:-1:1)], patchcm(2,:),'LineStyle','none');
end

h(1) = plot(x, m, 'LineWidth', 2, 'Color', linecm(1,:));
if (~isempty(pumpMs2))
    h(2) = plot(x, m2, 'LineWidth', 2, 'Color', linecm(2,:));
end
h(3) = plot([startDelay startDelay+stimLen], [yBar yBar], 'LineWidth', 16, 'Color', lightc);

t = title(['VIDEO: ' name ' (mean \pm sem, n = ' ...
            num2str(length(pumpMs)) ')'], 'FontWeight', 'bold', 'FontSize', 16);
if (~isempty(pumpMs2))
    t = title(['VIDEO: ' name ' (mean \pm sem, n = ' ...
            num2str(length(pumpMs2)) ')'], 'FontWeight', 'bold', 'FontSize', 16);
end
%set(t, 'interpreter', 'none');  % Turn off _ causing subscripts
xTitle = xlabel('time (s)', 'FontWeight', 'Bold', 'FontSize', 12);
yTitle = ylabel([eventName], 'FontWeight', 'Bold', 'FontSize', 12);
prettify();
ylim([yMin yMax]);
axis([xStart xStart+totalTime-winWidthInSec 0 yMax]);

%yts = 0:1:yMax;
%set(gca, 'YTick', yts); 

%xts = get(gca, 'XTick');
%xts = xts ./ 1000;
%set(gca, 'XTick', xts);

%legend(h, s1n, s2n, stimn);
stimText = text(startDelay+stimLen/2, yBar, '365 nm light',...
     'FontWeight', 'bold', 'FontSize', 10, 'Color', [1 1 1], ...
     'HorizontalAlignment', 'center');
%text(12.5, yMax-.35, stimn, 'FontWeight', 'bold');

%{
if(s1m(end) > s2m(end))
    y1 = s1m(end) + 0.4;
    y2 = s2m(end) - 0.6;
else
    y1 = s1m(end) - 0.6;
    y2 = s2m(end) + 0.4;
end
text(x(end), y1, s1n, ...
     'FontWeight', 'bold', 'FontSize', 24, 'Color', linecm(1,:), ...
     'HorizontalAlignment', 'right', 'VerticalAlignment', 'bottom');
text(x(end), y2, s2n, ...
     'FontWeight', 'bold', 'FontSize', 24, 'Color', linecm(2,:), ...
     'HorizontalAlignment', 'right', 'VerticalAlignment', 'bottom');
%}

if (forpaper) % Figure will be used for a paper so modify fonts and linewidths appropriately
    title('');
    set(h(3), 'LineWidth', 18);  % 20 for H2O2
    set(stimText, 'FontSize', 12);
    set(stimText, 'Position', [startDelay+stimLen/2 yBar+0.05]);
    set(stimText, 'FontName', 'Arial');
    set(stimText, 'FontWeight', 'normal');
    set(gca, 'YTickLabel', {'0','','','','','5',''});
    xlabels = {};
    oldxlabels = get(gca, 'XTickLabel');
    for i=1:length(oldxlabels)
        if(iscell(oldxlabels))
            if(mod(str2double(oldxlabels{i}), 10) == 0)
                xlabels{i} = oldxlabels{i};
            else
                xlabels{i} = '';
            end
        else
            if(mod(str2double(oldxlabels(i,:)), 10) == 0)
                xlabels{i} = oldxlabels(i,:);
            else
                xlabels{i} = '';
            end
        end
    end
    set(gca, 'XTickLabel', xlabels);
    set(gca, 'FontSize', 17);
    set(gca, 'FontName', 'Arial');
    set(xTitle, 'FontSize', 21);
    set(xTitle, 'FontName', 'Arial');
    set(xTitle, 'FontWeight', 'normal');
    set(yTitle, 'FontSize', 21);
    set(yTitle, 'FontName', 'Arial');
    set(yTitle, 'FontWeight', 'normal');    
    set(gca, 'LineWidth', 1.13);    
end

prettify();

%x = cell2mat(pumpRate);
%figure
%plot(x);
disp('Analysis complete.');

end